//MK40&MK64 IO subroutines
//V1.22 chage segment from 0 to 40 on peeks and pokes (needed for windows compile/run)
//
//V1.20 added getMKrevision to read and store MK's model# and software version & set num_of_inputs
//added new gobal variables

int verbose=0;			//verbose=show all steps;
int quiet=0;			//quiet=no details except errors.
int IOverbose=0;		//IOverbose=show IOsubsteps and/or other substeps.
int IOquiet=0;			//IOquiet=mask ALL output in this program.
int EEPROM[64];		//Stores MK EEPROM data 0 to 63. Not quirk compensated
int ack_time_limit=6;	//default is upto 60mS wait for ACK


unsigned long int Wait_more4ready=0;	//forces wait 4 ready sub to wait a fixed time (0=no delay)
char IOrevision[8]="V2.00";		//revision # of this code
char MKrevision[9]="00V0.00";		//note constant c++ address but data will change
int MKsoftware=0; //set by getMKrevision to 102 or 103 (valid values to date)
int num_of_inputs=0;//set by getMKrevision to 40 or 64 (valid values to date)

void write_port(char command);
	// sends a 8 bit word from the computer to the MKxx
	// this byte may be a command or a command parameter
	// there is currently no error checking at this level


void set_io_parameters(int argc,char* argv[]);
	// checks DOS argument for io setup default over rides.


int getEEPROM();
	// Pulls 64 bytes of EEPROM memory from the MK and stores in (int)EEPROM[64]
	//return: 0=Pass -1=timeout, -2=bad data

int getMKrevision();
	// Pulls model number and on board software revision from the MK
	// (int)num_of_inputs is set to 40 or 64
	// (int)MKsoftware is set to 100 * rev#, for example 103 means 1.03
	// (char*)MKrevision is ascii text, for example 40V1.03
	// return: 0=Pass -1=timeout, -2=bad data

void write_ram(char address,char data);
	// Writes a value (data) to a MKxx read/write memory location (address)
	// Can also be used to write to MKxx I/O ports and registers in bank zero
	// Refer to the memory map for more information
	// There is no error checking at this level

void write_high_ram(char address,char data);
	// Writes a value (data) to a MKxx read/write memory location (address)
	// Refer to the memory map for more information
	// There is no error checking at this level

int write_verify_ram(char address,char data,int time_limit,int mask=255);
	// Writes a value (data) to a MKxx read/write memory location (address)
	// Can also be used to write to MKxx I/O ports and registers in bank zero
	// Refer to the memory map for more information
	// Returns zero if successful, -1 on timeout err, -2 on data error
	// and -3 on verify error

int read_ram(char address,int time_limit);
	// Read the value (return) from a read/write memory location (address) inside the MKxx
	// Can also be used to read to MKxx I/O ports and registers in bank zero
	// Refer to the memory map for more information
	// Will retry upto 3 times as needed. (error correction)
	// Returns zero if successful, -1 on timeout err, -2 on data error.

void wait(int(time));
	// Waits for time * 10 miliseconds then returns

int comout_enable();
	// Allows the MKxx to pass keyboard and digital inputs to computer.
	// Enabled is the default
	// Returns zero if successful, -1 on error

int comout_disable();
	// Prevents the MKxx from passing keyboard and digital inputs to computer.
	// This is needed to pass commands to the MKxx with out interruption
	// Returns zero if successful, -1 on error
	// DO NOT LEAVE THE COMOUT DISABLED!

char getkey(int(time_limit));
	// Accept a processed key press from the MKxx via the computer's keyboard buffer
	// provided they arrive with in time * 10 miliseconds.
	// Special keys like ALT & CONTROL can not be read this way.
	// Returns zero (null) if timeout else returns data
//---------------------------------------------------------------------------------------

void wait_4_ready();
	// waits until the computer keyboard controller is ready to accept a 8 bit word.
	// this is already called by the other functions

long int get_time();
	// this is used by 'getkey' and 'wait' to read the system clock.

int single_pass_read_ram(char address,int time_limit);

//---------------------------------------------------------------------


void set_io_parameters(int argc,char* argv[])
{    int i,temp;
	for (i=1;i<argc;i++)

	{	if (argv[i][0]=='-')
		{	if((0x20|argv[i][1])=='v') //set verbose screen output
			    {verbose++;
				if ((0x20|argv[i][2])=='v') IOverbose++;
			    }
			if((0x20|argv[i][1])=='q') //set quiet screen output
			    {quiet++;
				if ((0x20|argv[i][2])=='q') IOquiet++;
			    }
			if((0x20|argv[i][1])=='a') //set fixed wait on wait4ready sub
			    {temp=argv[i][2]-48;  //ascii to decimal
				if ((temp>-1)&&(temp<10))
					ack_time_limit=temp; //0 to 9 is valid
			    }
		}
	}

	return;
}


int getEEPROM()		//alters EEPROM[64]
{	int i,value,time_limit=50;	//0.5 second
	char in,null=0x0;

	if (MKsoftware<103) wait(1);

	pokeb(0x40,0x17,peekb(0x40,0x17)&0xf0);//Turn shift,ctrl,and alt keys off!
	while(kbhit()) getch(); //get any stragglers in buffer
	pokeb(0x40,0x17,peekb(0x40,0x17)&0xf0);//Turn shift,ctrl,and alt keys off!

	if(verbose) cout <<"\nReading MK's EEPROM";
	write_port(0xa2);	//A2 (Output EEPROM data)
	in=getkey(time_limit);	//get first digit on model # (its OK if start char is missing)
	if (in==null) return -1;

	if ((in|0x20)!='x')
		{	if(!IOquiet) cout<<"\nIllegal start char#";
			return -2;
		}

	for(i=0;i<64;i++)
	{
		in=getkey(time_limit);	//get upper char
		if (in==null) return -1;	//timeout error
		in = (in |0x20);
		if (in>0x39) in-=39;	//convert letter A-F to 10-15
		value=16*(in-48);		//convert ascii to value
		if ((in<48)||(in>63)) return -2; 	//illegal char

		in=getkey(time_limit);
		if (in==null) return -1;	//timeout error
		in = (in |0x20);
		if (in>0x39) in-=39;	//convert letter A-F to 10-15
		value+=(in-48);			//convert ascii to value
		if ((in<48)||(in>63)) return -2;	//illegal char

		EEPROM[i]=value;
	}

	do
	{	in=getkey(time_limit);		//ignor copy right notice
		if (in==null) return -1;
	}
	while ((in|0x20)!='x');

	if (MKsoftware<103) wait(1);

	return 0;	//all passed
}


int getMKrevision()
{	int i,time_limit=50;	//0.5 second
	char in,null=0x0;

	pokeb(0x40,0x17,peekb(0x0,0x17)&0xf0);//Turn shift,ctrl,and alt keys off!
	while(kbhit()) getch(); //get any stragglers in buffer
	pokeb(0x40,0x17,peekb(0x40,0x17)&0xf0);//Turn shift,ctrl,and alt keys off!

	if(verbose) cout <<"\nReading MK's revision number";
	write_port(0xaf);	//AF (Output Revision and copyright)
	do { 	in=getkey(time_limit);	//get first digit on model # (its OK if start char is missing)
		if (in==null) return -1;
		}
	while(in=='\\');

	if ((in!='4')&&(in!='6')) {	if(!IOquiet) cout<<"\nIllegal Model#";
					return -2;
				  }
	MKrevision[0]=in;	//set first digit of model number

	in=getkey(time_limit);	//get second digit on model #
	if (in==null) return -1;
	if ((in!='0')&&(in!='4')) {	if(!IOquiet) cout<<"\nIllegal Model#";
					return -2;
				  }
	MKrevision[1]=in;	//set first digit of model number

	if ((MKrevision[0]=='4')&&(MKrevision[1]=='0')) num_of_inputs=40;
	if ((MKrevision[0]=='6')&&(MKrevision[1]=='4')) num_of_inputs=64;

	in=getkey(time_limit);	//get 'V'
	if (in==null) return -1;
	if ((in!='v')&&(in!='V')) {	if(!IOquiet) cout<<"\nIllegal REV";
					return -2;
				  }

	MKrevision[2]=in|0x20;	//set first digit of model number

	for(i=3;i<7;i++)
	{	in=getkey(time_limit);	//get second digit on model #
		if (in==null) return -1;
		if ( (in!='.')&&( (in<'0')||(in>'9') ) )
				{	if(!IOquiet) cout<<"\nIllegal version#";
					return -2;
				}
		MKrevision[i]=in;	//set first digit of model number
	}
	MKrevision[i]=0;	//set end string;
	if(verbose) cout << "\nMK revision is " << MKrevision;

	MKsoftware=(MKrevision[3]-48)*100;
	MKsoftware+=(MKrevision[5]-48)*10;
	MKsoftware+=(MKrevision[6]-48)*1;

	do
	{	in=getkey(time_limit);		//ignor copy right notice
		if (in==null) return -1;
	}
	while ((in|0x20)!='/');

	return 0;	//all passed
}



void write_port(char command)
{
if(IOverbose) cout <<"@";

	wait_4_ready();	//wait until keyboard controller is ready for a command

if(IOverbose) cout <<"#";

	poke(0x40,0x97,peek(0x40,0x97)&0xef);	//erase last ack from keyboard

	outportb(0x60,command);	//send command

	long int now,start_time=get_time();

	int i=0;

	if(ack_time_limit)	//wait x * 10mS for ack
	{
		do
		{ now=get_time();
			if (now<start_time)			//day rollover
				start_time-=8640000;	//subtract 1 day
			pokeb(0x40,0x17,peekb(0x40,0x17)&0xf0);
			i=peek(0x40,0x97)&0x10;
		}
		while ( (!i)&&(now<start_time+ack_time_limit) ); //ACK time limit
	}

if(IOverbose)
	{ if(i) cout <<"$";
	  else  cout <<"X";

	}

}

void wait_4_ready()
{    unsigned read;
	unsigned long int count;

	count=Wait_more4ready;		//fixed delay, the lamest way to fix timing.
	while (count>1) {count++;count--;count--;}

	do	read=inportb(0x64);	//get key port status
	while (read & 2);
}

void write_ram(char address,char data)
{
	if(IOverbose) cout <<"\nWriting "<<(int)data<<" into RAM location "<<(int)address<<"...";

	write_port(0xad);	//AD (ram write)

	write_port(address);	//ram address

	write_port((data&240)/16); //ram data upper nybble

	write_port(data&15);	//ram lower nybble

	if(IOverbose) cout <<"  Done";

	return;
}


void write_high_ram(char address,char data)
{
	if(IOverbose) cout <<"\nWriting "<<(int)data<<" into HIGH RAM location "<<(int)address<<"...";

	write_port(0xa3);	//AD (high ram write)

	write_port(address);	//ram address

	write_port((data&240)/16); //ram data upper nybble

	write_port(data&15);	//ram lower nybble

	if(IOverbose) cout <<"  Done";

	return;
}


int write_verify_ram(char address,char data,int time_limit,int mask)
{	if(IOverbose) cout<<"\nStarting a RAM write verify";
	int r_data,i;
	int error=-3;
	write_ram(address,data);
	r_data=read_ram(address,time_limit);
	if (r_data<0) {error=r_data;}
	if (r_data>-1)
		{	if (!( (r_data^data) & mask)) {error=0;}
		}

	if(IOverbose)
		{	if (error) cout<<"\nWrite verify failed";
			else       cout<<"\nWrite verify Passed";
		}
	return error;
}


int read_ram(char address,int time_limit)
{    int value;
	for(int i=0;i<3;i++)
	{	if(IOverbose) cout<<"\nReading RAM location "<<(int)address<<"..";
		value=single_pass_read_ram(address,time_limit);
		if (value>-1) break;
		wait(5);
		if(IOverbose) cout << "..Retry..";
	}
	if(IOverbose)
		{	if(value>-1) cout<<"Data is "<<value<<" ";
			else         cout<<"Failed";
		}
	return value;
}


int single_pass_read_ram(char address,int time_limit)
{    const int null=0x0;
	char in;

if (MKsoftware<103) wait(1);

	int value;
	write_port(0xac);	//AC (ram read)
	pokeb(0x40,0x17,peekb(0x40,0x17)&0xf0);//Turn shift,ctrl,and alt keys off!
	while(kbhit()) getch(); //get any stragglers in buffer
	pokeb(0x40,0x17,peekb(0x40,0x17)&0xf0);//Turn shift,ctrl,and alt keys off!

	write_port((char)address);	//AE (EEPROM address)
	in=getkey(time_limit);	//get upper char
	if (in==null) return -1;	//timeout error
	in = (in |0x20);
	if (in>0x39) in-=39;	//convert letter A-F to 10-15
	value=16*(in-48);		//convert ascii to value
	if ((in<48)||(in>63)) return -2; 	//illegal char

	in=getkey(time_limit);
	if (in==null) return -1;	//timeout error
	in = (in |0x20);
	if (in>0x39) in-=39;	//convert letter A-F to 10-15
	value+=(in-48);			//convert ascii to value
	if ((in<48)||(in>63)) return -2;	//illegal char

if (MKsoftware<103) wait(1);
	return value;
}


long int get_time()
{    //this returns the current time in .01 sec units
	union REGS regs;	 //copied from book pg.397
	regs.h.ah = 0x2c;	 //get time from register clock
	intdos(&regs,&regs); //copied from book pg.397
	return (regs.h.dl+100*regs.h.dh+6000*regs.h.cl+360000*regs.h.ch);
}

void wait(int(time))
{    long int start_time=get_time();
	long int now;

	if(IOverbose) cout<<":";

	do{	now=get_time();
		if (now<start_time)			//day rollover
			start_time-=8640000;	//subtract 1 day
	  }
	while (get_time()<start_time+time);


	return;
}


char getkey(int(time_limit))
{    const char null=0x0;
	char key;
	long int now,start_time=get_time();

	if(IOverbose) cout <<" ?";
	key=null;
	do
	{ now=get_time();
		if (now<start_time)			//day rollover
			start_time-=8640000;	//subtract 1 day
		pokeb(0x40,0x17,peekb(0x40,0x17)&0xf0);
		if (kbhit()) key=getch();
	}
	while ( (key==null)&&(now<start_time+time_limit) );
	if(IOverbose)
	{	if(key==null) cout <<"Getkey Timeout";
		else	      cout <<"!";
		return key;
	}
	if((!IOquiet)&&(key==null)) cout <<"\nGetkey Timeout error";
	return key;
}

int comout_disable()
{	const char comstat=16,comout_dis=4;
	int i,error=-1;
	if(verbose) cout <<"\nDisabling Comout..";

	for(i=0;i<5;i++)
	{
	write_ram(comstat,comout_dis);///value|comout_dis);	//turn comout off
	while (kbhit()) getch();
	pokeb(0x40,0x17,peekb(0x40,0x17)&0xf0);
	wait(5);
	while (kbhit()) getch();
	if(4==read_ram(16,50)) {error=0;break;}
	if(verbose) cout << "Retry...";
	wait(5);
	}
	if(verbose)
	{	if(error) cout <<"..Failed";
		else      cout <<"..Done";
	}
	if (error) write_ram(comstat,0);//try to turn comout back on
	return error;
}

int comout_enable()
{	const char comstat=16;
	int error,value;
	if(verbose) cout <<"\nEnabling Comout..";

	for (int i=0;i<10;i++)	//try up to 10 times to enable comout
	{    error=-1;
		write_ram(comstat,0);	//turn comout on
		while (kbhit()) getch();
		pokeb(0x40,0x17,peekb(0x40,0x17)&0xf0);
		wait(5);
		while (kbhit()) getch();

		value=read_ram(comstat,50);
		if (value==0 )
			{	error=0;break;	}
		if(verbose) cout << "Retry...";
	}

	if(verbose)
	{	if(error) {cout <<"..Unverified";return error;}
		else      cout <<"..Done";
	}

	if (error) cout << "\nWarning - Comout may still be disabled!!!";
	return error;
}
